// **********************************************
//	Fence LED control program with UNO R4 WiFi
// **********************************************
//	Ver1.0	2025/05/29

#include <arduino.h>
//#include <stdio.h>
//#include <stdint.h>
#include <Arduino_FreeRTOS.h>
#include <EEPROM.h>
#include <wire.h>
#include "WiFiS3.h"
#include "arduino_secrets.h" 
#include "FenceLedControl.h"

extern "C" {
	#include "xprintf.h"
}

WiFiUDP		Udp;
WiFiServer	server(80);
SemaphoreHandle_t semBuf=NULL;

// *** Debug/Setting enable/disable ***
//#define	ADDRSET		// MCUにアドレスセットを行う

// declaration
void printWifiStatus();
void SC8721_begin();
extern void ledPatternMakeTask(void *pvParameters);
extern void resetCounter();

const int NTP_PACKET_SIZE = 48; 	// NTP timestamp is in the first 48 bytes of the message
byte 	packetBuffer[NTP_PACKET_SIZE]; //buffer to hold incoming and outgoing packets
char	ssid[] = SECRET_SSID;		// network SSID (name)
char	pass[] = SECRET_PASS;		// network password (use for WPA, or use as key for WEP)
int		keyIndex = 0;				// network key index number (needed only for WEP)
int		status = WL_IDLE_STATUS;
volatile uint8_t	patSelNo = 99;
volatile uint8_t	tFlag = 0;		// 1:Serial1 Ch data trans enable flag
volatile uint8_t	trMode = 0;		// 3 or 5 byte trans mode flag
volatile uint16_t	thAmbient = 10;	// 周囲光（暗さ）しきい値
volatile uint16_t	ambient;		// ambient data
volatile uint16_t	retryVal = 0;	// NPT retry number
volatile uint32_t	utcVal = 0;		// RTC value (UTC:second)
volatile int		cnt8hour=0;		// 周囲光制御用の１秒カウンタ
volatile int		ntpCnt=NTPACS;
volatile int	 	ambStat=3;	// 周囲光制御のステートマシンパラメータ
volatile dt4byte	ledbuf[32]={0};	// ledバッファ
volatile dt4byte	ledbufb[32]={0};// ledバッファ(前回値)
volatile uint16_t	corrBright=127;	// 明るさ係数（4時間以降に徐々に暗くする）
uint16_t			adamb[BUFSIZE]={0};	// for moving ave

const uint8_t encode_table[128] = {	// 7B8B符号化テーブル(分解能は125までなので、残りの3個は255で埋める)
   37, 38, 39, 41, 42, 43, 45, 46, 47, 50, 51, 53, 54, 55, 57, 58, 59, 61, 62, 63, 73, 74, 75, 77, 78, 79, 82, 83, 85, 86, 87, 89,
   90, 91, 93, 94, 95,101,102,103,105,106,107,109,110,111,114,115,117,118,119,121,122,123,125,126,127,146,147,149,150,151,153,154,
  155,157,158,159,165,166,167,169,170,171,173,174,175,178,179,181,182,183,185,186,187,189,190,191,201,202,203,205,206,207,210,211,
  213,214,215,217,218,219,221,222,223,229,230,231,233,234,235,237,238,239,242,243,245,246,247,249,250,251,253,254,255,255,255,255
};

// 7bitデータを7B8B変換
uint8_t encode_7b8b(uint8_t in7) {
	return encode_table[in7 & 0x7F];
}

void	semTake() {
	xSemaphoreTake(semBuf,portMAX_DELAY);
}

void	semGive() {
	xSemaphoreGive(semBuf);
}

// LEDバッファをチェックし、更新したLEDバッファのデータ送信(約1msごとに右左に振分け)
void ledDataSendTask(void *pvParameters) {
	uint8_t	dt[5];
	uint8_t	addr = 0;
	dt4byte	ledpat;

	vTaskDelay(1000);
	while(1) {
		digitalWrite(MON,1);
		semTake();
		ledpat.d32 = ledbuf[addr].d32;
		semGive();
		// 立上り時１秒以外は前回バッファと異なるNoのLEDに順次送信、アドレス書き込みモードではOff
		if(((tFlag==0) && (ledbuf[addr].d32 != ledbufb[addr].d32)) || (cnt8hour<2)) {
			uint8_t	leng = 3;
			if(trMode==0) {
				leng = 5;
				dt[0] = encode_7b8b(addr);		// LEDアドレス(ロングコードアドレス:0x0-0x1f)
				dt[1] = encode_7b8b(ledpat.BR);	// 7B8B符号化した BR 輝度データ
				dt[2] = encode_7b8b(ledpat.R);	// 7B8B符号化した R カラーデータ
				dt[3] = encode_7b8b(ledpat.G);	// 7B8B符号化した G カラーデータ
				dt[4] = encode_7b8b(ledpat.B);	// 7B8B符号化した B カラーデータ
			} else {
				dt[0] = encode_7b8b(addr | 0x40);	// LEDアドレス(ショートコードアドレス:0x40-0x5f)
				dt[1] = encode_7b8b(ledpat.BR);	// 7B8B符号化した BR 輝度データ
				dt[2] = encode_7b8b(ledpat.COL);// 7B8B符号化した R カラーデータ
			}
			if((addr&0x10)!=0) {				// 左右のドライブ信号振り分け(ch0-15:左、ch16-31:右)
				digitalWrite(LEDDIR1,1);		// left Line select
				digitalWrite(LEDDIR2,0);
			} else {
				digitalWrite(LEDDIR1,0);		// right Line select
				digitalWrite(LEDDIR2,1);
			}
			for(int i=0;i<leng;i++) Serial1.write(dt[i]);	// UART送信
			ledbufb[addr].d32 = ledpat.d32;		// 比較用の前回値bufferへコピー
		}
		addr = ++addr & 0x1f;
		digitalWrite(MON,0);
		vTaskDelay(1);
	}
}

uint8_t	address = 0x62;
int	setvol = 730;	// 7.3V * 100

// DC-DC SC8721 initialize 7.3V output
void SC8721_begin() {
	int MSB = ((setvol-500)/2) >> 2;
	int LSB = ((setvol-500)/2) & 0x3;
	Wire.beginTransmission(address);
	Wire.write(0x3);		// regaddr:03
	Wire.write(MSB);		// reg03:0x64>>2
	Wire.write(0x18+LSB);	// reg04:0x18+0x64&mask(0x03)
	Wire.write(0x02);		// reg05:load
	Wire.write(0x80);		// reg06:I2C active
	Wire.endTransmission();	//送信を完了
}

// DC-DC SC8721 initialize 0V output
void SC8721_off() {
	Wire.beginTransmission(address);
	Wire.write(0x5);		// regaddr:05 select
	Wire.write(0x04);		// reg05:DCDC disable
	Wire.endTransmission();	//送信を完了
}

#ifdef	ADDRSET	// この機能は廃止
// 設定用のディップスイッチリード
uint8_t readDSW() {
	uint8_t value = 0;
	value |= digitalRead(D4);	// D4
	value |= (digitalRead(D5) << 1); // D5
	value |= (digitalRead(D6) << 2); // D6
	value |= (digitalRead(D7) << 3); // D7
	value |= (digitalRead(D8) << 4); // D8
	value |= (digitalRead(D9) << 5); // D9
	value |= (digitalRead(D10) << 6); // D10
	value |= (digitalRead(D11) << 7); // D11
	return ~value;
}

// LEDのMCUにdipswのb0-b4のアドレスを書き込むタスク(全配線後は全部変更してしまうので厳禁)
void settingTask(void *pvParameters) {
	char s[32];
	unsigned char	dt[4];

	while(1) {
		if(digitalRead(D11)==0) {	// DIPSW 8 on?
			if(digitalRead(PUSW)==0) {
				digitalWrite(MON,1);
				tFlag = 1;	// 通常データ送信禁止
				SC8721_off();
				while(digitalRead(PUSW)==0) vTaskDelay(20);	// send 2times
				SC8721_begin();
				vTaskDelay(500);
				uint8_t dsw = (readDSW() & 0x3f);
				xsprintf(s,"\r\n%x\r\n", dsw);
				Serial.print(s);
				dt[0] = 0xff;
				dt[1] = encode_7b8b(dsw);		// ch data send
				dt[2] = 0xff;
				dt[3] = 0xff;
				dt[4] = 0xff;
				for(int i=0;i<4;i++) Serial1.write(dt[i]);	// send
				vTaskDelay(200);
				for(int i=0;i<4;i++) Serial1.write(dt[i]);
				vTaskDelay(500);
				tFlag = 0;	// 通常データ送信禁止解除
				digitalWrite(MON,0);
			}
		}
		vTaskDelay(100);
	}
}
#endif

#include "htmlMessage.html"	//　WEBサーバーで使うHTMLの文字列

void loop() {}	// UNO R4でFreeRTOSを使うとloop()には飛ばない(らしい)

unsigned long sendNTPpacket() {
	memset(packetBuffer, 0, NTP_PACKET_SIZE);
	packetBuffer[0] = 0b11100011;	// LI, Version, Mode
	Udp.beginPacket("2.jp.pool.ntp.org", 123);	//NTP requests are to port 123
	Udp.write(packetBuffer, NTP_PACKET_SIZE);
	Udp.endPacket();
}

// メインタスク(loop()の代わり)
// Webサーバー処理、周囲光測定、周囲光によるon/off動作を行う
void mainTask(void *pvParameters) {
	int		ambcnt=0;	// 移動平均バッファのpointer格納
	unsigned long startTime=0;	// Timeクラスを使って、1秒計測するための変数
	int 	ambTotal=0;	// 移動平均用トータル格納変数
	char	strg[128];	// xsprintf用の文字列バッファ（完成後は削除）
	int		ambStartCnt=0;
	int		tempAve = 0,tempVal = 0;
	uint16_t	disNtp = 0;

	startTime = millis();
	uint16_t ambdata = analogRead(A0);		// 移動平均領域を立上り周囲光で初期化
	for(int i=0;i<BUFSIZE;i++) adamb[i]=ambdata;
	ambTotal = ambdata*BUFSIZE;
	while(1) {
		WiFiClient client = server.available();		// listen for incoming clients
		int cnt = 0;
		if (client) {								// if you get a client,
//Serial.println("new client");			// print a message out the serial port
			String currentLine = "";				// make a String to hold incoming data from the client
			while (client.connected()) {			// loop while the client's connected
				if (client.available()) {			// if there's bytes to read from the client,
					char c = client.read();			// read a byte, then
					Serial.write(c);				// print it out to the serial monitor
					if (c == '\n') {				// if the byte is a newline character
						if (currentLine.length() == 0) {	// blank & 2xCR, send  responce
							client.println("HTTP/1.1 200 OK");	// responce code
							client.println("Content-type:text/html");
							client.println();
							client.print(htmlMessage1);	// send HTML code
							client.print("閾値:");
							client.print(thAmbient);
							client.print(" STAT:");
							client.print(ambStat);
							client.print(" retry:");
							client.print(retryVal);
							client.print(" 補正値:");
							client.println(corrBright);
							client.print("<br>");
							client.print("温度:");
							client.print(tempVal);
							client.print("  現在時刻:");
							client.print((((utcVal % 86400) / 3600)+9) % 24 );
							client.print(":");
							client.println((utcVal % 3600) / 60);
							client.print(htmlMessage2);	// send HTML code
							break;					// break out of the while loop:
						} else {					// if you got a newline, then clear currentLine:
							currentLine = "";
						}
						cnt++;
						vTaskDelay(2);
					} else if (c != '\r') {			// if you got anything else but a carriage return character,
						currentLine += c;			// add it to the end of the currentLine
					}
					// クライアントリクエストをエンドポイントをチェックして操作
					if (currentLine.endsWith("GET /pat1")) {
						resetCounter();
						patSelNo = 1;
						EEPROM.write(0,1);		// 点灯パターン1:EEPROMアドレス0に格納
					}
					if (currentLine.endsWith("GET /pat2")) {
						resetCounter();
						patSelNo = 2;
						EEPROM.write(0,2);		// 点灯パターン2:EEPROMアドレス0に格納
					}
					if (currentLine.endsWith("GET /pat3")) {
						resetCounter();
						patSelNo = 3;
						EEPROM.write(0,3);		// 点灯パターン3:EEPROMアドレス0に格納
					}
					if (currentLine.endsWith("GET /pat4")) {
						resetCounter();
						patSelNo = 4;
						EEPROM.write(0,4);		// 点灯パターン4:EEPROMアドレス0に格納
					}
					if (currentLine.endsWith("GET /pat5")) {
						resetCounter();
						patSelNo = 5;
						EEPROM.write(0,5);		// 点灯パターン5:EEPROMアドレス0に格納
					}
					if (currentLine.endsWith("GET /pat6")) {
						resetCounter();
						patSelNo = 6;
						EEPROM.write(0,6);		// 点灯パターン6:EEPROMアドレス0に格納
					}
					if (currentLine.endsWith("GET /pat7")) {
						resetCounter();
						patSelNo = 7;
						EEPROM.write(0,7);		// 点灯パターン7:EEPROMアドレス0に格納
					}
					if (currentLine.endsWith("GET /pat8")) {
						resetCounter();
						patSelNo = 8;
						EEPROM.write(0,8);		// 点灯パターン8:EEPROMアドレス0に格納
					}
					if (currentLine.endsWith("GET /pat99")) {
						resetCounter();
						patSelNo = 99;
						EEPROM.write(0,99);		// 点灯パターン99:EEPROMアドレス0に格納
					}
					if (currentLine.endsWith("GET /advalue")) {
						String ambStr = String(ambient);			// 周囲光のAD値
						String response = "HTTP/1.1 200 OK\r\n";	// HTTP/1.1形式のレスポンスにして返す
						response += "Content-Type: text/plain\r\n";	// Content-Type: プレーンテキスト（HTMLではない）
						response += "Content-Length: " + String(ambStr.length()) + "\r\n";	// レスポンスの長さ
						response += "Connection: close\r\n\r\n";	// コネクションを閉じる
						response += ambStr;
						client.print(response);	// クライアントに送信
//						client.print(ambient);	// これだけだとHTTP/0.9になってしまう
						break;
					}
					if (currentLine.indexOf("POST /setThreshold") >= 0) {	// 受信したしきい値を読み出す
						String postData = "";
						while (client.available()) {	// 全受信文字列読み出し
							char c = client.read();
							postData += c;
						}
						int index = postData.indexOf("th=");// POSTデータの中から"th="に続く数値を探し変換
						if (index >= 0) {					// あった！
							String valueStr = postData.substring(index + 3);
							valueStr.trim();
							int value = valueStr.toInt();
							if (valueStr.length() > 0 && value >= 0 && value <= 255) {	// 数値かチェック
								thAmbient = (uint16_t)value;
								EEPROM.write(2,(uint8_t)(thAmbient & 0xff));	// 照度しきい値セーブ
							}
						}
						client.println("HTTP/1.1 303 See Other");	// ホームに戻す簡易レスポンス返す
						client.println("Location: /");
						client.println();
						break;
					}
				}
//				vTaskDelay(1);	// ここではディレイ置くと文字受信の間にディレイが入ってしまう
			}
			client.stop();	// close the connection:
			Serial.println();
		}
		if(millis()-startTime>=SECOND1) {	// 1s毎に行う処理(unsigned longなのでオーバーフローOK)
			startTime=millis();
			// 周囲光測定（BUFSIZEの移動平均）
			ambdata = analogRead(A0);
			ambTotal = ambTotal+ambdata-adamb[ambcnt];// 移動平均処理（Total=Total-古い値+新しい値）
			adamb[ambcnt] = ambdata;		// リングバッファに新しい値格納
			ambient = ambTotal/BUFSIZE/20;	// 最大値51
			if(++ambcnt >= BUFSIZE) ambcnt=0;
			uint16_t thAmbH=thAmbient*5/2;	// 明けのレベルは薄暮レベルの2.5倍とする
			// 温度測定
			tempAve = (tempAve + analogRead(A1))/2;	// 2 level moving average
			tempVal = (int)(map(tempAve, 0, 1023, 0, 5060) - 424) / 6.25;	// 5060は実測値5.06V
			switch (ambStat) {	// 周囲光点灯制御のステートマシン
				case 0:	// 消灯状態(日中状態or点灯してから8時間経過後の状態)
					if(ambient<thAmbient) {	// 暗くなった？
						corrBright = 127;	// 明るさ係数：max
						cnt8hour=0;			// 時間カウントスタート
						SC8721_begin();		// DC-DC Power ON
						ambStat=1;
					}
					break;
				case 1:		// 点灯状態 -> ONして4時間経過以降は徐々に暗くする
					if(ambient>thAmbH) {	// 周囲が明るくなった？
						SC8721_off();		// DC-DC Power OFF
						ambStat=0; corrBright = 127;
					}
					if((utcVal>HOUR23) && (utcVal<=HOUR27)) {	// 23:00～03:00までは徐々に明るさダウンさせる
						corrBright=127-(utcVal-HOUR23)/113;
					} else corrBright=127;
					if(utcVal>HOUR27) {		// 午前3時過ぎたら完全消灯
						SC8721_off();		// DC-DC Power OFF
						ambStat=2;
					}
					break;
				case 2:	// 午前3時以降で完全消灯して再び明るくなるまで待つ
					if(ambient>thAmbH) {	// 周囲が明るくなった？
						ambStat=0; corrBright = 127;
					}
					break;
				case 3:	// 起動開始
					if(++ambStartCnt>2) {	// 2秒待つ
						if(ambient<thAmbient) {
//							SC8721_begin();
							ambStat=0;
						}
					}
					break;
				default : ambStat=0; break;
			}
			cnt8hour++;
			ntpCnt = utcVal % 21600L;	// 時刻校正は8時間に一回
			if(((ntpCnt==0) && (disNtp>600)) || (ambStartCnt==1)) {	// 起動時と6時間ごとにNTPサーバーから読み出しUTCでutcValへ保存（ただし前回校正から10分間以上経過）
				sendNTPpacket();
				retryVal = 99;	// 校正失敗したらリトライ回数は99とする
				for(uint8_t i=0;i<20;i++) {	// 20msはレスポンス待ってみる
					if(Udp.parsePacket()) {
						Udp.read(packetBuffer, NTP_PACKET_SIZE);	// 時刻(Transmit Timestamp)の取得
						unsigned long transTime = packetBuffer[40]<<24 | packetBuffer[41]<<16 | packetBuffer[42]<<8 | packetBuffer[43];
						utcVal = transTime % 86400L;	// UTCのままにしておく
						retryVal = i;
						Serial.println("time correct");
						break;
					}
					vTaskDelay(10);
				}
				disNtp = 0;	// 校正OK/NGに拘わらず、4時間以上経過した6時間を待つ
			}
if((utcVal % 10)==0) {
	xsprintf(strg,"%u %d %d %d %d %d %d %d",utcVal,corrBright,ambient,thAmbient,thAmbH,ambStat,tempVal,tempAve); Serial.println(strg);
}
			disNtp++;
			utcVal = ++utcVal % 86400L;
//xsprintf(strg,"%ud %d %d %d %d %d",corrBright,cnt8hour,ambient,ambStat,ambTotal,thAmbient);
//Serial.println(strg);
		}
		vTaskDelay(10);
	}
}

// Output WiFi Status to monitor
void printWifiStatus() {
	Serial.print("SSID: ");			// print the SSID of the network you're attached to:
	Serial.println(WiFi.SSID());
	IPAddress ip = WiFi.localIP();	// print your board's IP address:
	Serial.print("IP Address: ");
	Serial.println(ip);
	long rssi = WiFi.RSSI();		// print the received signal strength:
	Serial.print("signal strength (RSSI):");
	Serial.print(rssi);
	Serial.println(" dBm");
	Serial.print("To see this page in action, open a browser to http://");	// print where to go in a browser:
	Serial.println(ip);
}

void setup() {
	Serial.begin(115200);			// initialize serial communication
	Serial1.begin(57600);
	srand((unsigned int)time(NULL)); // 乱数の種を初期化
	Wire.begin();
//	EEPROM.write(0,99);				// テスト用の99書き込み
	patSelNo = EEPROM.read(0);		// 点灯パターン:EEPROMアドレス0に格納
	thAmbient = (uint16_t)EEPROM.read(2);	// 照度しきい値読み出し
	SC8721_off();					// DC-DCをOFFにしておく（念のため）
	pinMode(LED_BUILTIN, OUTPUT);	// set the LED pin mode
	pinMode(MON, OUTPUT);			// set the MON pin mode
	pinMode(LEDDIR1,OUTPUT);
	pinMode(LEDDIR2,OUTPUT);
	pinMode(D4,INPUT_PULLUP);
	pinMode(D5,INPUT_PULLUP);
	pinMode(D6,INPUT_PULLUP);
	pinMode(D7,INPUT_PULLUP);
	pinMode(D8,INPUT_PULLUP);
	pinMode(D9,INPUT_PULLUP);
	pinMode(D10,INPUT_PULLUP);
	pinMode(D11,INPUT_PULLUP);
	pinMode(PUSW,INPUT_PULLUP);
//	analogReadResolution(10);		// default 10bit
	digitalWrite(LEDDIR1,1);		// OUT1,2をHighにしておき、SC8721を立ち上げる
	digitalWrite(LEDDIR2,1);
	// WiFi setup
	if (WiFi.status() == WL_NO_MODULE) {// check for the WiFi module:
		Serial.println("Communication with WiFi module failed!");
		while (true);				// don't continue
	}
	String fv = WiFi.firmwareVersion();
	if (fv < WIFI_FIRMWARE_LATEST_VERSION) {
		Serial.println("Please upgrade the firmware");
	}
	while (status != WL_CONNECTED) {// attempt to connect to WiFi network:
		Serial.print("Attempting to connect to Network named: ");
		Serial.println(ssid);		// print the network name (SSID);
		status = WiFi.begin(ssid, pass);// Connect to WPA/WPA2 network. Change this line if using open or WEP network:
		delay(5000);				// wait 5 seconds for connection:
	}
	server.begin();					// start the web server on port 80
	Udp.begin(2390);
	printWifiStatus();				// you're connected now, so print out the status
//	SC8721_begin();					// DC-DC on
	delay(10);
	semBuf = xSemaphoreCreateBinary();
	xSemaphoreGive(semBuf);
 	xTaskCreate(mainTask, "mainTask", 512, NULL, 4, NULL);
	xTaskCreate(ledDataSendTask, "ledDataSendTask", 256, NULL, 4, NULL);	// バッファ更新されたらLEDに信号出力するタスク
	xTaskCreate(ledPatternMakeTask, "ledPatternMakeTask", 512, NULL, 4, NULL);
#ifdef	ADDRSET		// アドレス書き込みタスク(通常は未使用)
	xTaskCreate(settingTask, "setting", 128, NULL, 4, NULL);
#endif
	Serial.println("setup complete");
	vTaskStartScheduler();
	for(;;);
}
